## function to map degree binned impacts for CONUS
map_impacts <- function(
    df         = NULL,                  # df with AT LEAST columns fips (FIPS codes as strings of length appropriate for the spatial scale, e.g. "01", "01021"), model (gcm names), degree (integer degrees), value (data to be mapped); if sector has multiple impactTypes, see additional parameter below
    geography  = NULL,                  # spatial scale of data, one of ["block_group", "tract", "county", "state"]
    sector     = NULL,                  # sector name to use for plots and file names
    impactType = FALSE,                 # set to TRUE if sector has multiple impactTypes, and df passed to the function must include a column called impactType
    label      = "incidence per capita" # how to label the scale key
){
  packages <- c("tidyverse", "tigris", "mapview", "sf")
  for(package in packages){
    if(!(package %in% installed.packages()[, "Package"])) {
      paste0("Installing ", package, " package...") %>% message
      install.packages(package, dependencies = TRUE)
    }
  }
  library(tidyverse, quietly = TRUE)
  library(tigris,    quietly = TRUE)
  library(mapview,   quietly = TRUE)
  library(sf,        quietly = TRUE)
  
  # average across models
  if(impactType){
    df <- df %>%
      group_by(fips, degree, impactType) %>%
      summarize(value = mean(value), .groups = "drop") %>%
      mutate(degree = paste0(degree, " Degree"),
             degree = case_when(
               degree == "0 Degree" ~ "Baseline",
               TRUE ~ degree
             )) %>%
      mutate(degree = factor(degree,
                             levels = c("Baseline", paste0(1:6, " Degree")) %>%
                               intersect(unique(degree)))) %>%
      filter(!fips %in% c("02", "15", "60", "66", "69", "72", "78"))
  } else{
    df <- df %>%
      group_by(fips, degree) %>%
      summarize(value = mean(value), .groups = "drop") %>%
      mutate(degree = paste0(degree, " Degree"),
             degree = case_when(
               degree == "0 Degree" ~ "Baseline",
               TRUE ~ degree
             )) %>%
      mutate(degree = factor(degree,
                             levels = c("Baseline", paste0(1:6, " Degree")) %>%
                               intersect(unique(degree)))) %>%
      filter(!fips %in% c("02", "15", "60", "66", "69", "72", "78"))  
  }

  # store a state outline object for use in plots
  us_outline <- states(year = 2020, cb = TRUE) %>% filter(!STATEFP %in% c("02", "15", "60", "66", "69", "72", "78"))
  
  # grab spatial data for the appropriate geography
  if(geography == "block_group") spatial <- block_groups(state = NULL, year = 2020, cb = TRUE)
  if(geography == "tract")       spatial <-       tracts(state = NULL, year = 2020, cb = TRUE)
  if(geography == "county")      spatial <-     counties(state = NULL, year = 2020, cb = TRUE)
  if(geography == "state")       spatial <-       states(              year = 2020, cb = TRUE)

  # join spatial data and impact data
  geo_df <- spatial %>%
    left_join(df, by = c("GEOID" = "fips")) %>%
    filter(!is.na(value))
  
  # create plot object
  custom_colors <- colorRampPalette(c("white", "yellow", "orange", "red", "darkred"))(5)
  impact_plot <- ggplot(geo_df) +
    geom_sf(aes(fill = value), color = NA) + # plot impact data
    geom_sf(data = us_outline, fill = NA, color = "black", size = 0.5) + # plot state outlines
    scale_fill_gradientn(colors = custom_colors) + # shade according to impact data using color scale defined above
    theme_minimal() +
    theme(
      panel.grid      = element_blank(), # Remove grid lines
      axis.title      = element_blank(), # Remove axis titles
      axis.text       = element_blank(), # Remove axis text
      axis.ticks      = element_blank(), # Remove axis ticks
      plot.background = element_rect(fill = "transparent", color = NA),     # Transparent background
      plot.title      = element_text(hjust = 0.5, size = 16, face = "bold") # Centered title with style
    ) +
    labs(title = sector,
         fill  = label)
  
  # add facet grid if plotting multiple impactTypes
  if(impactType){
    impact_plot <- impact_plot + facet_grid(degree ~ impactType)
  } else{
    impact_plot <- impact_plot + facet_wrap(~ degree, ncol = 2)
  }
  
  # save plot object to png
  ggsave(paste0(sector, "_plots.png"), plot = impact_plot, width = 10, height = 8, dpi = 300)
  return(geo_df)
}