

; Processing commercial satellite imagery, Draft version 13
; 10 August 2020


; Written by Megan M. Coffer
;   ORISE Fellow, U.S. Environmental Protection Agency, Office of Research and Development, Durham, NC
;   PhD Candidate, Center for Geospatial Analytics, North Carolina State University, Raleigh, NC


; Permission is granted for an individual or institution to use this script provided that any 
; publications developed using this script properly cite the manuscript accompanying this script:
; Coffer, M. M., Schaeffer, B. A., Zimmerman, R. C, Hill, V., Li, J., Islam, K. A. & Whitman, 
;   P. J. (2020). Performance across WorldView-2 and RapidEye for reproducible seagrass mapping. 
;   Remote Sensing of Environment. DOI: 10.1016/j.rse.2020.112036. 



; Known issues
;   1) This code was built for Worldview-2 and Worldview-3 processing only. RapidEye processing 
;      is implemented in a separate script.
;   2) Sometimes ENVI might freeze and say 'Not responding,' but it usualy is still responding. 
;      Give the script plenty of time to process.
;   3) Panchromatic imagery is not included.
;   4) Deleting temporary files still generates an error which could not be resolved even with 
;      the assistance of Harris Geospatial. 


; Define the IDL code with a single input variable, aoi, which indicates the string name of the 
; area of interest. The aoi variable should follow the exact filename of the folder containing 
; the files for the area of interest (i.e. punctuation, underscores, etc.).
PRO process_wv_20200810, aoi


  ; Compile IDL
  COMPILE_OPT IDL2


  ; Define the Rayleigh exponent to use for atmospheric correction
  rayleighExp = 4.75


  ; Define AOI and change working directory
  PRINT, 'Processing satellite imagery for ' + aoi
  rootDir = 'E:\Estuary_Seagrass\WorldView_Imagery\'
  workingDir = rootDir + aoi


  ; Suppress automated console messages
  !quiet = 1


  ; List zip files in 1_Zip_File
  zipFiles = FILE_SEARCH(workingDir + '\1_Zip_File\', '*.zip', /FULLY_QUALIFY_PATH)


  ; Loop through each file in zipFiles
  FOR i = 0, N_ELEMENTS(zipFiles) - 1 DO BEGIN


    ; Initiate ENVI session
    e = ENVI()
    
    
    ; Create and set a temporary folder
    prefs = e.Preferences
    tempFolder = FILEPATH('tempFolder\', ROOT_DIR = workingDir)
    FILE_MKDIR, tempFolder
    prefs['directories and files:temporary directory'].VALUE = tempFolder
    
    
    ; extentRoi outlines the extent of the area of interest
    extentRoiPath = FILEPATH('InputData\extents\' + aoi + '_extent.xml', ROOT_DIR = rootDir)
    extentRoi = e.OpenRoi(extentRoiPath)
    ; estuariesRoi outlines the estuary for the area of interest 
    estuariesRoiPath = FILEPATH('InputData\estuaries\' + aoi + '_estuaries.xml', ROOT_DIR = rootDir)
    estuariesRoi = e.OpenRoi(estuariesRoiPath)
    ; demInput is a global DEM raster file
    demFile = FILEPATH('InputData\GMTED2010.jp2', ROOT_DIR = rootDir)
    demInput = e.OpenRaster(demFile)


    ; Subset to the ith zip file
    zipFile = zipFiles[i]
    fileBasename = FILE_BASENAME(zipFile, '.zip')
    ; Unzip the ith zip file into 2_Raw_Data
    PRINT, STRING(9B) + 'Unzipping raw imagery for file: ' + STRCOMPRESS(fileBasename + ' (File ' + STRING(i + 1) + ' of ' + STRING(N_ELEMENTS(zipFiles)) + ')')
    IF ~ FILE_TEST(FILEPATH('2_Raw_Data/' + fileBasename + '_rawData', ROOT_DIR = workingDir)) THEN BEGIN
      ; Create a string for the output folder
      unzipFile = FILEPATH('2_Raw_Data/' + fileBasename + '_rawData', ROOT_DIR = workingDir)
      ; Create a string for the SPAWN command input
      spawnInput = "powershell.exe Expand-Archive -Path " + zipFile + " -DestinationPath " + unzipFile + " -Verbose"
      ; Call powershell from IDL to unzip the
      SPAWN, spawnInput
    ENDIF


    ; List .TIL files for the current scene
    satFiles = FILE_SEARCH(FILEPATH('2_Raw_Data/' + fileBasename + '_rawData', ROOT_DIR = workingDir), '*M1BS*' + '*.TIL', /FULLY_QUALIFY_PATH)


    ; Define an array to populate with red edge values for each tile
    rededgeList = MAKE_ARRAY(N_ELEMENTS(satFiles), 1, /FLOAT, VALUE = -9999)
    ; Define arrays to populate with image information
    tileFilename = MAKE_ARRAY(N_ELEMENTS(satFiles), 1, /STRING, VALUE = 'NA')
    tileViewAngle = MAKE_ARRAY(N_ELEMENTS(satFiles), 1, /FLOAT, VALUE = -9999)
    tileSunAngle = MAKE_ARRAY(N_ELEMENTS(satFiles), 1, /FLOAT, VALUE = -9999)
    tileREdgeAnchor = MAKE_ARRAY(N_ELEMENTS(satFiles), 1, /FLOAT, VALUE = -9999)
    tileRayleigh = MAKE_ARRAY(N_ELEMENTS(satFiles), 1, /FLOAT, VALUE = -9999)
    tileProcessingTime = MAKE_ARRAY(N_ELEMENTS(satFiles), 1, /STRING, VALUE = 'NA')


    ; Loop through each tile, read it in, and process
    FOR j = 0, N_ELEMENTS(satFiles) - 1 DO BEGIN


      ; Print progress of the loop
      PRINT, STRING(9B) + STRING(9B) + 'Processing steps 1 through 7 for tile' + STRCOMPRESS(STRING(j + 1) + ' of ' + STRING(N_ELEMENTS(satFiles)))
      ; Print progress of the loop
      PRINT, STRING(9B) + STRING(9B) + STRING(9B) + STRCOMPRESS('Reading in raster as a WorldView scene (Step 1 of 10)')


      ; Read in .TIL image
      file = satFiles[j]
      raster = e.OpenRaster(file)
      ; Define outputFile for saving rasters
      outputFile = STRCOMPRESS(fileBasename + '_' + STRING(j + 1), /REMOVE_ALL)
      ; Determine if the tile is within the estuariesRoi extent
      rasterRoiExtent = estuariesRoi.GetExtent(raster)


      ; If raster does not fall within the extent of estuariesRoi, do not continue processing this file
      IF(rasterRoiExtent EQ !NULL) THEN BEGIN
        
        
        ; Print a statement that the tile will not be processed
        PRINT, STRING(9B) + STRING(9B) + 'This tile will not be processed as it does not fall within the ROI for the given estuary.'


      ; If raster does fall within the extent of estuariesRoi, continue processing.
      ENDIF ELSE BEGIN

           
        ; Check if the tile covers at least 20% of extentRoi before continuing processing
        ; Determine the number of pixels in extentRoi that intersect demInput
        pixelsInROI = extentRoi.PixelCount(demInput)
        ; Find the area of these pixels assuming each DEM pixel is approximately 900 m, which is sufficient for the current task
        areaInROI = pixelsInROI * (900 ^ 2)
        ; Determine the number of pixels in extentRoi that intersect raster
        pixelsOverlap = extentRoi.PixelCount(raster)
        ; Find the area of these pixels based on a spatial resolution of 2 m
        areaOverlap = pixelsOverlap * (2 ^ 2)
        ; Find the percent overlap between areaInROI and areaOverlap
        pctOverlap = FLOAT(areaOverlap) / FLOAT(areaInROI)


        ; If the scene covers less than 20% of the ROI, do not continue processing
        IF(pctOverlap LT 0.2) THEN BEGIN

          ; Print a statement that the tile will not be processed
          PRINT, STRING(9B) + STRING(9B) + 'This tile will not be processed as it covers less than 20% of the ROI for the given estuary.'

        
        ; If raster covers more than 20% of the extent of estuariesRoi, continue processing.
        ENDIF ELSE BEGIN
          

          ; Print a statement that the tile will be processed
          PRINT, STRING(9B) + 'This tile will be cropped to the ROI for the given estuary and processed.'
          ; Crop by extentRoi to reduce processing
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Masking and cropping scene to match AOI extent (Step 2 of 10)'
          rasterCropped = ENVIROIMaskRaster(raster, extentRoi)
  
  
          ; Update metadata and perform radiometric calibration
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Applying radiometric calibration (Step 3 of 10)'
          IF ~ FILE_TEST(FILEPATH('3_Rad_Cal/' + outputFile + '_radCal/' + outputFile + '_radCal.til', ROOT_DIR = workingDir)) THEN BEGIN
            ; Define the task
            Task_RadCal = ENVITask('RadiometricCalibration')
            ; Open input raster
            Task_RadCal.INPUT_RASTER = rasterCropped
            ; Add data ignore value
            metadata = rasterCropped.Metadata
            IF (metadata.HasTag ('DATA IGNORE VALUE')) THEN BEGIN
              metadata.UpdateItem, 'DATA IGNORE VALUE', 999999
            ENDIF ELSE BEGIN
              metadata.AddItem, 'DATA IGNORE VALUE', 999999
            ENDELSE
            ; Extract absolute calibration factor and effective bandwidth for the scene
            imd_files = FILE_SEARCH(FILEPATH('2_Raw_Data/' + fileBasename + '_rawData', ROOT_DIR = workingDir), '*M1BS*' + '*.IMD')
            imd_file = imd_files[j]
            ; Read in metadata
            OPENR, unit, imd_file, /GET_LUN
              array = ''
              line = ''
              WHILE NOT EOF(unit) DO BEGIN & $
                READF, unit, line & $
                array = [array, line] & $
              ENDWHILE
            FREE_LUN, unit
            ; Extract the absolute calibration factor and effective bandwidth for each band
            absCal = FLOAT(STRMID(array[WHERE(STRMATCH(array, '*absCalFactor*'))],16,17))
            effBW = FLOAT(STRMID(array[WHERE(STRMATCH(array, '*effectiveBandwidth*'))],22,23))
            ; Adjust gains and offsets according to the sensor
            IF STRMATCH(metadata['SENSOR TYPE'], 'WorldView-2') THEN BEGIN
              newScaleFactors = [1.151, 0.988, 0.936, 0.949, 0.952, 0.974, 0.961, 1.002]
              metadata.UpdateItem, 'DATA GAIN VALUES', newScaleFactors * (absCal / effBW)
              metadata.UpdateItem, 'DATA OFFSET VALUES', [-7.478, -5.736, -3.546, -3.564, -2.512, -4.120, -3.300, -2.891]
            ENDIF ELSE IF STRMATCH(metadata['SENSOR TYPE'], 'WorldView-3') THEN BEGIN
              newScaleFactors = [0.905, 0.940, 0.938, 0.962, 0.964, 1.000, 0.961, 0.978]
              metadata.UpdateItem, 'DATA GAIN VALUES', newScaleFactors * (absCal / effBW)
              metadata.UpdateItem, 'DATA OFFSET VALUES', [-8.604, -5.809, -4.996, -3.649, -3.021, -4.521, -5.522, -2.992]
            ENDIF
            ; Set parameters for radiometric calibration and run task
            Task_RadCal.CALIBRATION_TYPE = 'Top-of-Atmosphere Reflectance'
            Task_RadCal.Execute
            ; Output results into new folder
            radCalOutput = Task_RadCal.OUTPUT_RASTER
            FILE_MKDIR, FILEPATH('3_Rad_Cal/' + outputFile + '_radCal', ROOT_DIR = workingDir)
            radCalOutput.Export, FILEPATH('3_Rad_Cal/' + outputFile + '_radCal/' + outputFile + '_radCal.til', ROOT_DIR = workingDir), 'ENVI'
          ENDIF
    
    
          ; Divide all pixel values across all bands by pi
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Converting R to Rrs (Step 4 of 10)'
          IF ~ FILE_TEST(FILEPATH('4_R_To_Rrs/' + outputFile + '_rToRrs/' + outputFile + '_rToRrs.til', ROOT_DIR = workingDir)) THEN BEGIN
            rrsRasterFile = workingDir + '/3_Rad_Cal/' + outputFile + '_radCal/' + outputFile + '_radCal.til'
            rrsRaster = e.OpenRaster(rrsRasterFile)
            ; Define an empty dictionary and an empty list to populate with results for each band
            rrsAggregator = Dictionary()
            rrsList = List()
            ; Extract each band from rrsRaster
            Task_ExtractBands = ENVITask('ExtractBandsFromRaster')
            Task_ExtractBands.INPUT_RASTER = rrsRaster
            Task_ExtractBands.Execute
            ; Loop through each band and apply band math
            FOREACH k, Task_ExtractBands.OUTPUT_RASTERS, k_index DO BEGIN
              ; Apply band math to divide all pixel values by pi
              Task_BandMath = ENVITask('PixelwiseBandMathRaster')
              Task_BandMath.INPUT_RASTER = k
              Task_BandMath.EXPRESSION = 'b1 / !pi'
              Task_BandMath.Execute
              ; Add results to rrsList
              rrsList.Add, Task_BandMath.OUTPUT_RASTER, /EXTRACT
              rrsAggregator.OUTPUT = rrsList
            ENDFOREACH
            ; Stack each band back together using BuildBandStack
            Task_Stack = ENVITask('BuildBandStack')
            Task_Stack.INPUT_RASTERS = rrsAggregator.OUTPUT
            Task_Stack.OUTPUT_RASTER_URI = "*"
            Task_Stack.Execute
            bandMathOutput = Task_Stack.OUTPUT_RASTER
            ; Transfer metadata from rrsRaster to bandMathOutput
            FOR metadataTag = 0, N_ELEMENTS(rrsRaster.Metadata.Tags) - 1 DO BEGIN
              IF bandMathOutput.Metadata.HasTag (rrsRaster.Metadata.Tags[metadataTag]) THEN BEGIN
                bandMathOutput.Metadata.UpdateItem, rrsRaster.Metadata.Tags[metadataTag], rrsRaster.Metadata[rrsRaster.Metadata.Tags[metadataTag]]
              ENDIF ELSE BEGIN
                bandMathOutput.Metadata.AddItem, rrsRaster.Metadata.Tags[metadataTag], rrsRaster.Metadata[rrsRaster.Metadata.Tags[metadataTag]]
              ENDELSE
            END
            ; Output results into new folder
            FILE_MKDIR, FILEPATH('4_R_To_Rrs/' + outputFile + '_rToRrs/', ROOT_DIR = workingDir)
            bandMathOutput.Export, FILEPATH('4_R_To_Rrs/' + outputFile + '_rToRrs/' + outputFile + '_rToRrs.til', ROOT_DIR = workingDir), 'ENVI'
          ENDIF
    
    
          ; Get just water pixels by masking the raster by our ROI to subset spatially and masking by NDWI values to subset spectrally
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Masking raster image to just water pixels (Step 5 of 10)'
          IF ~ FILE_TEST(FILEPATH('5_Mask_Raster/' + outputFile + '_maskRaster/' + outputFile + '_maskRaster.til', ROOT_DIR = workingDir)) THEN BEGIN
            shpMaskRasterFile = FILEPATH('4_R_To_Rrs/' + outputFile + '_rToRrs/' + outputFile + '_rToRrs.til', ROOT_DIR = workingDir)
            shpMaskRasterInput = e.OpenRaster(shpMaskRasterFile)
            shpMaskRaster = ENVIROIMaskRaster(shpMaskRasterInput, estuariesRoi)
            ; Calculate NDWI on shpMaskRaster
            Task_NDWI = ENVITask('PixelwiseBandMathRaster')
            Task_NDWI.INPUT_RASTER = shpMaskRaster
            Task_NDWI.EXPRESSION = 'float(b3 - b7) / float(b3 + b7)'
            Task_NDWI.Execute
            rasterNDWI = Task_NDWI.OUTPUT_RASTER
            ; Create a mask raster based on an NDVI threshold
            ndwiThreshold = [0]
            ndwiThresholdRaster = ENVIBinaryLTThresholdRaster(rasterNDWI, ndwiThreshold)
            ; Apply ndwiThresholdRaster as a threshold mask on shpMaskRaster
            Task_Mask = ENVITask('MaskRaster')
            Task_Mask.DATA_IGNORE_VALUE = 999999
            Task_Mask.INPUT_MASK_RASTER = ndwiThresholdRaster
            Task_Mask.INPUT_RASTER = shpMaskRaster
            Task_Mask.INVERSE = 'TRUE'
            Task_Mask.Execute
            ; Output results into new folder
            maskOutput = Task_Mask.OUTPUT_RASTER
            FILE_MKDIR, FILEPATH('5_Mask_Raster/' + outputFile + '_maskRaster/', ROOT_DIR = workingDir)
            maskOutput.Export, FILEPATH('5_Mask_Raster/' + outputFile + '_maskRaster/' + outputFile + '_maskRaster.til', ROOT_DIR = workingDir), 'ENVI'
          ENDIF
    
    
          ; Extract band six (red edge) values and save to list
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Computing atmospheric correction reference value (Step 6 of 10)'
          ; Read in results from 5_Mask_Raster
          redEdgeFile = FILEPATH('5_Mask_Raster/' + outputFile + '_maskRaster/' + outputFile + '_maskRaster.til', ROOT_DIR = workingDir)
          redEdgeInput = e.OpenRaster(redEdgeFile)
          ; Subset to red edge values and get data greater than 999999 (our data ignore value)
          redEdgeValues = redEdgeInput.GetData(BANDS = [5])
          redEdgeValuesSubset = redEdgeValues[WHERE(redEdgeValues NE redEdgeInput.Metadata['DATA IGNORE VALUE'])]
          redEdgeValuesSort = redEdgeValuesSubset[SORT(redEdgeValuesSubset)]
          ; Retrieve half of the lowest 5% of the data within the red edge
          redEdgeAnchor = MEDIAN(redEdgeValuesSort[0:(N_ELEMENTS(redEdgeValuesSubset) / 20) - 1]) / 2
          
          
          ; Get information about the scene and populate arrays
          imd_files = FILE_SEARCH(FILEPATH('2_Raw_Data/' + fileBasename + '_rawData', ROOT_DIR = workingDir), '*M1BS*' + '*.IMD')
          imd_file = imd_files[j]
          ; Read in metadata
          OPENR, unit, imd_file, /GET_LUN
            array = ''
            line = ''
            WHILE NOT EOF(unit) DO BEGIN & $
              READF, unit, line & $
              array = [array, line] & $
            ENDWHILE
          FREE_LUN, unit
          ; Extract the meanOffNadirViewAngle and meanSunEl for the scene
          tileFilename[j] = outputFile
          tileViewAngle[j] = FLOAT(STRMID(array[WHERE(STRMATCH(array, '*meanOffNadirViewAngle*'))],24,27))
          tileSunAngle[j] = FLOAT(STRMID(array[WHERE(STRMATCH(array, '*meanSunEl*'))],13,16))
          tileREdgeAnchor[j] = redEdgeAnchor
          tileRayleigh[j] = rayleighExp
          tileProcessingTime[j] = SYSTIME()
  
        ENDELSE
  
      ENDELSE
  
    ENDFOR

    ; Save scene information as a CSV file
    PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Outputing file information as a CSV file (Step 7 of 10)'
    IF ~ FILE_TEST(FILEPATH('File_Information/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '_fileInformation.csv', ROOT_DIR = workingDir)) THEN BEGIN
      ; Create a CSV file of the results
      FILE_MKDIR, FILEPATH('File_Information/', ROOT_DIR = workingDir)
      fileinfo_output = FILEPATH('File_Information/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '_fileInformation.csv', ROOT_DIR = workingDir)
      WRITE_CSV, fileinfo_output, tileFilename, tileViewAngle, tileSunAngle, tileREdgeAnchor, tileRayleigh, tileProcessingTime, $
        HEADER = ['FILENAME', 'VIEWANGLE', 'SUNANGLE', 'REDEDGEANCHOR', 'RAYLEIGH', 'PROCESSINGTIME']
    ENDIF
    
  
    ; Find the minimum rededge value for the entire overpass and compute the atmospheric correction values
    ; Determine which satellite is being processed to define wavelengths
    wvFilename = FILE_BASENAME(zipFiles[i], '.zip')
    sensorName = STRMID(wvFilename,STRPOS(wvFilename, 'WV'), 3)
    IF(sensorName EQ 'WV2') THEN BEGIN
      wavelengths = [427.3,477.9,546.2,607.8,658.8,723.7,832.5,908.0]
    ENDIF ELSE BEGIN
      wavelengths = [425.0,480.0,545.0,605.0,660.0,725.0,832.5,950.0]
    ENDELSE
    ; Find the minimum value from rededgeList and compute atmospheric correction values
    minRedEdgeAnchor = MIN(tileREdgeAnchor[WHERE(tileREdgeAnchor NE -9999)])
    ; Compute the scattering factor based on the minRededge value
    scatteringFactor = wavelengths[5] ^ rayleighExp * minRedEdgeAnchor
    ; Use the scatteringFactor to compute the correction values for each band
    atmCorrValues = MAKE_ARRAY(N_ELEMENTS(wavelengths), 1, /FLOAT, VALUE = 999999)
    FOR j = 0, N_ELEMENTS(wavelengths) - 1 DO BEGIN
      atmCorrValues[j] = scatteringFactor / wavelengths[j] ^ rayleighExp
    ENDFOR
  
  
    ; List processed files from 4_R_To_Rrs for further processing 
    rToRrs_files = FILE_SEARCH(workingDir + '\4_R_To_Rrs\', '*.TIL')


    ; Loop through each tile in rToRrs_files and apply additional processing 
    FOR j = 0, N_ELEMENTS(rToRrs_files) - 1 DO BEGIN
      
      
      ; Update the progress of the loop
      PRINT, STRING(9B) + STRING(9B) + 'Processing step 8 through 10 for tile' + STRCOMPRESS(STRING(j + 1) + ' of ' + STRING(N_ELEMENTS(rToRrs_files)))
    
    
      ; Define outputFile
      outputFile = STRMID(FILE_BASENAME(rToRrs_files[j],'.TIL'),0,STRLEN(FILE_BASENAME(rToRrs_files[j],'.TIL'))-7)
      
    
      ; Apply atmospheric correction
      PRINT, STRING(9B) + STRING(9B) + STRING(9B) + STRCOMPRESS('Applying atmospheric correction (Step 8 of 10)')
      IF ~ FILE_TEST(FILEPATH('6_Atm_Corr/' + outputFile + '_atmCorr/' + outputFile + '_atmCorr.til', ROOT_DIR = workingDir)) THEN BEGIN
        ; Read in raster results from 4_R_To_Rrs as atmospheric correction input
        atmCorrFile = rToRrs_files[j]
        atmCorrInput = e.OpenRaster(atmCorrFile[0])
        ; Apply the DarkSubtractionCorrection task
        Task_atmCorr = ENVITask('DarkSubtractionCorrection')
        Task_atmCorr.INPUT_RASTER = atmCorrInput
        Task_atmCorr.VALUES = atmCorrValues
        Task_atmCorr.Execute
        ; Output results into new folder
        atmCorrOutput = Task_atmCorr.OUTPUT_RASTER
        FILE_MKDIR, FILEPATH('6_Atm_Corr/' + outputFile + '_atmCorr/', ROOT_DIR = workingDir)
        atmCorrOutput.Export, FILEPATH('6_Atm_Corr/' + outputFile + '_atmCorr/' + outputFile + '_atmCorr.til', ROOT_DIR = workingDir), 'ENVI'
        atmCorrOutput.Export, FILEPATH('6_Atm_Corr/' + outputFile + '_atmCorr/' + outputFile + '_atmCorr.tif', ROOT_DIR = workingDir), 'TIFF'
      ENDIF
    
    
;      ; OPTIONAL -- Apply RPCOrthorectification
;      PRINT, STRING(9B) + STRING(9B) + STRING(9B) + STRCOMPRESS('Applying RPC Orthorectification (Step 9 of 10)')
;      IF ~ FILE_TEST(FILEPATH('7_RPC_Ortho/' + outputFile + '_rpcOrtho/' + outputFile + '_rpcOrtho.til', ROOT_DIR = workingDir)) THEN BEGIN
;        ; Read in results from atmospheric correction
;        rpcOrthoFile = FILEPATH('6_Atm_Corr/' + outputFile + '_atmCorr/' + outputFile + '_atmCorr.til', ROOT_DIR = workingDir)
;        rpcOrthoInput = e.OpenRaster(rpcOrthoFile, DATA_IGNORE_VALUE = 0.0)
;        ; Read in global DEM
;        demFile = FILEPATH('InputData\GMTED2010.jp2', ROOT_DIR = rootDir)
;        demInput = e.OpenRaster(demFile)
;        ; Apply the RPCOrthorectification task
;        Task_rpcOrtho = ENVITask('RPCOrthorectification')
;        Task_rpcOrtho.INPUT_RASTER = rpcOrthoInput
;        Task_rpcOrtho.DEM_RASTER = demInput
;        Task_rpcOrtho.OUTPUT_PIXEL_SIZE = [2.0,2.0]
;        Task_rpcOrtho.Execute
;        ; Output results into new folder
;        rpcOrthoOutput = Task_rpcOrtho.OUTPUT_RASTER
;        FILE_MKDIR, FILEPATH('7_RPC_Ortho/' + outputFile + '_rpcOrtho/', ROOT_DIR = workingDir)
;        rpcOrthoOutput.Export, FILEPATH('7_RPC_Ortho/' + outputFile + '_rpcOrtho/' + outputFile + '_rpcOrtho.til', ROOT_DIR = workingDir), 'ENVI'
;      ENDIF
      
    ENDFOR
    
    
;    ; OPTIONAL -- Apply mosaicing
;    PRINT, STRING(9B) + STRING(9B) + STRCOMPRESS('Mosaicing tiles into single scene (Step 10 of 10)')
;    IF ~ FILE_TEST(FILEPATH('8_Mosaic/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '_mosaic/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '_mosaic.til', ROOT_DIR = workingDir)) THEN BEGIN
;      ; List RPC Orthorectified rasters
;      rpcOrthoList = FILE_SEARCH(FILEPATH('7_RPC_Ortho/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '*_rpcOrtho', ROOT_DIR = workingDir), '*rpcOrtho.til', /FULLY_QUALIFY_PATH)
;      ; Create an empty list to populate with rasters
;      rpcOrthoRasters = List()
;      ; Loop through each raster, read it in, and add to rpcOrthoRasters
;      FOR k = 0, N_ELEMENTS(rpcOrthoList) - 1 DO BEGIN
;        rpcOrthoRaster = e.OpenRaster(rpcOrthoList[k])
;        rpcOrthoRasters.Add, rpcOrthoRaster
;      ENDFOR
;      ; Convert rpcOrthoRasters to an array
;      mosaicInputRasters = rpcOrthoRasters.ToArray()
;      ; Apply the BuildMosaicRaster task
;      Task_mosaic = ENVITask('BuildMosaicRaster')
;      Task_mosaic.INPUT_RASTERS = mosaicInputRasters
;      Task_mosaic.Execute
;      ; Output results into new folder
;      mosaicOutput = Task_mosaic.OUTPUT_RASTER
;      FILE_MKDIR, FILEPATH('8_Mosaic/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '_mosaic/', ROOT_DIR = workingDir)
;      mosaicOutput.Export, FILEPATH('8_Mosaic/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '_mosaic/' + STRMID(outputFile, 0, STRLEN(outputFile) - 2) + '_mosaic.til', ROOT_DIR = workingDir), 'ENVI'
;    ENDIF
    
  ENDFOR
  
  ; Close ENVI and delete temporary folder
  e.Close
  ;FILE_DELETE, tempFolder, /RECURSIVE

END

