

; Processing commercial satellite imagery, Draft version 13
; 10 August 2020


; Written by Megan M. Coffer
;   ORISE Fellow, U.S. Environmental Protection Agency, Office of Research and Development, Durham, NC
;   PhD Candidate, Center for Geospatial Analytics, North Carolina State University, Raleigh, NC


; Permission is granted for an individual or institution to use this script provided that any 
; publications developed using this script properly cite the manuscript accompanying this script:
; Coffer, M. M., Schaeffer, B. A., Zimmerman, R. C, Hill, V., Li, J., Islam, K. A. & Whitman, 
;   P. J. (2020). Performance across WorldView-2 and RapidEye for reproducible seagrass mapping. 
;   Remote Sensing of Environment. DOI: 10.1016/j.rse.2020.112036. 



; Known issues
;   1) This code was built for Worldview-2 and Worldview-3 processing only. RapidEye processing 
;      is implemented in a separate script.
;   2) Sometimes ENVI might freeze and say 'Not responding,' but it usualy is still responding. 
;      Give the script plenty of time to process.
;   3) Panchromatic imagery is not included.
;   4) Deleting temporary files still generates an error which could not be resolved even with 
;      the assistance of Harris Geospatial. 


; Define the IDL code with a single input variable, aoi, which indicates the string name of the 
; area of interest. The aoi variable should follow the exact filename of the folder containing 
; the files for the area of interest (i.e. punctuation, underscores, etc.).
PRO process_re_20200810, aoi


  ; Compile IDL
  COMPILE_OPT IDL2


  ; Define the Rayleigh exponent to use for atmospheric correction
  rayleighExp = 4.75


  ; Define root directory (rootDir) and working directory (workingDir)
  PRINT, 'Processing satellite imagery for ' + aoi
  rootDir = 'E:\Estuary_Seagrass\RapidEye_Imagery\'
  workingDir = rootDir + aoi


  ; Suppress automated console messages
  !quiet = 1


  ; List rawDataFiles for the given AOI
  rawDataFiles = FILE_SEARCH(workingDir + '\2_Raw_Data\', '*_rawData')
  ; Determine the basename of each file in rawDataFiles
  rawDataBasename = FILE_BASENAME(rawDataFiles)
  ; Find the unique number of characters in each element in rawDataBasename
  nCharacters = STRLEN(rawDataBasename)
  uniqueCharacters = nCharacters[UNIQ(nCharacters)]
  ; Determine the collection dates for each filename
  rawDataDates = STRMID(rawDataBasename,uniqueCharacters-18,10)
  ; Find unique dates in rawDataDates
  uniqueDates = rawDataDates[UNIQ(rawDataDates)]


  ; Loop through each date in uniqueDates
  FOR i = 0, N_ELEMENTS(uniqueDates) - 1 DO BEGIN


    ; Initiate ENVI session
    e = ENVI()


    ; Create and set a temporary folder
    prefs = e.Preferences
    tempFolder = FILEPATH('tempFolder\', ROOT_DIR = workingDir)
    FILE_MKDIR, tempFolder
    prefs['directories and files:temporary directory'].VALUE = tempFolder


    ; Read in required input files
    ; extentRoi outlines the extent of the area of interest
    extentRoiPath = FILEPATH('InputData\extents\' + aoi + '_extent.xml', ROOT_DIR = rootDir)
    extentRoi = e.OpenRoi(extentRoiPath)
    ; estuariesRoi outlines the estuary for the area of interest
    estuariesRoiPath = FILEPATH('InputData\estuaries\' + aoi + '_estuaries.xml', ROOT_DIR = rootDir)
    estuariesRoi = e.OpenRoi(estuariesRoiPath)
    ; demInput is a global DEM raster file
    demFile = FILEPATH('InputData\GMTED2010.jp2', ROOT_DIR = rootDir)
    demInput = e.OpenRaster(demFile)


    ; Subset to the ith rawData file
    dateRawData = FILE_SEARCH(workingDir + '\2_Raw_Data\' + aoi + '_RE_' + uniqueDates[i] + '_rawData/' + '*metadata.xml')
    ; Print progress of the loop
    PRINT, STRING(9B) + 'Processing imagery for date: ' + STRCOMPRESS(uniqueDates[i] + ' (File ' + STRING(i + 1) + ' of ' + STRING(N_ELEMENTS(uniqueDates)) + ')')


    ; Define an array to populate with red edge values for each tile
    rededgeList = MAKE_ARRAY(N_ELEMENTS(dateRawData), 1, /FLOAT, VALUE = -9999)
    ; Define arrays to populate with image information
    tileFilename = MAKE_ARRAY(N_ELEMENTS(dateRawData), 1, /STRING, VALUE = 'NA')
    tileViewAngle = MAKE_ARRAY(N_ELEMENTS(dateRawData), 1, /FLOAT, VALUE = -9999)
    tileSunAngle = MAKE_ARRAY(N_ELEMENTS(dateRawData), 1, /FLOAT, VALUE = -9999)
    tileREdgeAnchor = MAKE_ARRAY(N_ELEMENTS(dateRawData), 1, /FLOAT, VALUE = -9999)
    tileRayleigh = MAKE_ARRAY(N_ELEMENTS(dateRawData), 1, /FLOAT, VALUE = -9999)
    tileProcessingTime = MAKE_ARRAY(N_ELEMENTS(dateRawData), 1, /STRING, VALUE = 'NA')


    ; Loop through each tile, read it in, and process
    FOR j = 0, N_ELEMENTS(dateRawData) - 1 DO BEGIN


      ; Print progress of the loop
      PRINT, STRING(9B) + STRING(9B) + 'Processing steps 1 through 7 for tile' + STRCOMPRESS(STRING(j + 1) + ' of ' + STRING(N_ELEMENTS(dateRawData)))
      PRINT, STRING(9B) + STRING(9B) + STRING(9B) + STRCOMPRESS('Reading in raster as a RapidEye scene (Step 1 of 10)')


      ; Determine filename for 2_Raw_Data
      rawDataFilename = STRCOMPRESS(aoi + '_RE_' + uniqueDates[i])
      ; List .XML files and .JSON files
      satFile = FILE_SEARCH(FILEPATH('2_Raw_Data/' + rawDataFilename + '_rawData', ROOT_DIR = workingDir), '*metadata.xml', /FULLY_QUALIFY_PATH)
      metaFiles = FILE_SEARCH(FILEPATH('2_Raw_Data/' + rawDataFilename + '_rawData', ROOT_DIR = workingDir), '*metadata.json', /FULLY_QUALIFY_PATH)


      ; Read in metadata via .JSON file for the given tile and extract time information
      OPENR, metaFile, metaFiles[j], /GET_LUN
      metaInfo = ''
      metaLine = ''
      WHILE NOT EOF(metaFile) DO BEGIN & $
        READF, metaFile, metaLine & $
        metaInfo = [metaInfo, metaLine] & $
      ENDWHILE
      FREE_LUN, metaFile


      ; Extract the acquisition time from the metadata file
      acqTime = STRMID(metaInfo[1], STREGEX(metaInfo[1], 'acquired') + 12, 19) + 'Z'
  
  
      ; Read in the raster from satFile using acqTime as the acquisition time and setting the sensor as RapidEye
      raster = e.OpenRaster(satFile[j], TIME = EnviTime(ACQUISITION = acqTime), DATASET_NAME = 'RapidEye')
      ; Determine the file basename by subsetting uniqueDates
      fileBasename = STRCOMPRESS(aoi + '_RE_' + uniqueDates[i] + '_' + STRING(j + 1), /REMOVE_ALL)
      ; Determine if the tile is within extentRoi
      rasterRoiExtent = extentRoi.GetExtent(raster)
  
  
      ; If raster does not fall within extentRoi, do not continue processing this file
      IF(rasterRoiExtent EQ !NULL) THEN BEGIN
  
  
        ; Print a statement that the tile will not be processed
        PRINT, STRING(9B) + STRING(9B) + 'This tile will not be processed as it does not fall within the ROI for the given estuary.'
  
  
        ; If raster does fall within the extent of estuariesRoi, continue processing.
      ENDIF ELSE BEGIN


        ; Check if the tile covers at least 20% of extentRoi before continuing processing
        ; Determine the number of pixels in extentRoi that intersect demInput
        pixelsInROI = extentRoi.PixelCount(demInput)
        ; Find the area of these pixels assuming each DEM pixel is approximately 900 m, which is sufficient for the current task
        areaInROI = pixelsInROI * (900 ^ 2)
        ; Determine the number of pixels in extentRoi that intersect raster
        pixelsOverlap = extentRoi.PixelCount(raster)
        ; Find the area of these pixels based on a spatial resolution of 6.5 m
        areaOverlap = pixelsOverlap * (6.5 ^ 2)
        ; Find the percent overlap between areaInROI and areaOverlap
        pctOverlap = FLOAT(areaOverlap) / FLOAT(areaInROI)
  
  
        ; If the scene covers less than 20% of the ROI, do not continue processing
        IF(pctOverlap LT 0.2) THEN BEGIN
  
  
          ; Print a statement that the tile will not be processed
          PRINT, STRING(9B) + STRING(9B) + 'This tile will not be processed as it covers less than 20% of the ROI for the given estuary.'
  
  
        ; If raster covers more than 20% of the extent of estuariesRoi, continue processing.
        ENDIF ELSE BEGIN


          ; Print a statement that the tile will be processed
          PRINT, STRING(9B) + STRING(9B) + 'This tile will be cropped to the ROI for the given estuary and processed.'
          ; Crop by extentRoi to reduce processing
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Masking and cropping scene to match AOI extent (Step 2 of 10)'
          rasterCropped = ENVIROIMaskRaster(raster, extentRoi)
  
  
          ; Update sun elevation metadata for rasterCropped
          IF rasterCropped.Metadata.HasTag ('SUN ELEVATION') THEN BEGIN
            rasterCropped.metadata.UpdateItem, 'SUN ELEVATION', STRMID(metaInfo[1], STREGEX(metaInfo[1], 'sun_elevation') + 15, 13)
          ENDIF ELSE BEGIN
            rasterCropped.metadata.AddItem, 'SUN ELEVATION', STRMID(metaInfo[1], STREGEX(metaInfo[1], 'sun_elevation') + 15, 13)
          ENDELSE
          ; Update data ignore value metadata for rasterCropped
          IF rasterCropped.Metadata.HasTag ('DATA IGNORE VALUE') THEN BEGIN
            rasterCropped.metadata.UpdateItem, 'DATA IGNORE VALUE', 999999.0
          ENDIF ELSE BEGIN
            rasterCropped.metadata.AddItem, 'DATA IGNORE VALUE', 999999.0
          ENDELSE
  
  
          ; Apply radiometric calibration to convert to TOA Reflectance
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Applying radiometric calibration (Step 3 of 10)'
          IF ~ FILE_TEST(FILEPATH('3_Rad_Cal/' + fileBasename + '_radCal/' + fileBasename + '_radCal.til', ROOT_DIR = workingDir)) THEN BEGIN
            ; Call the RadiometricCalibration ENVITask
            Task_RadCal = ENVITask('RadiometricCalibration')
            Task_RadCal.INPUT_RASTER = rasterCropped
            Task_RadCal.CALIBRATION_TYPE = 'Top-of-Atmosphere Reflectance'
            Task_RadCal.Execute
            ; Output results into new folder
            radCalOutput = Task_RadCal.OUTPUT_RASTER
            FILE_MKDIR, FILEPATH('3_Rad_Cal/' + fileBasename + '_radCal', ROOT_DIR = workingDir)
            radCalOutput.Export, FILEPATH('3_Rad_Cal/' + fileBasename + '_radCal/' + fileBasename + '_radCal.til', ROOT_DIR = workingDir), 'ENVI'
          ENDIF
  
  
          ; Convert from R to normalized water leaving reflectance by dividing all values of each band by pi
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Converting R to Rrs (Step 4 of 10)'
          IF ~ FILE_TEST(FILEPATH('4_R_To_Rrs/' + fileBasename + '_rToRrs/' + fileBasename + '_rToRrs.til', ROOT_DIR = workingDir)) THEN BEGIN
            ; Read in radiometric calibration result
            rrsRasterFile = workingDir + '/3_Rad_Cal/' + fileBasename + '_radCal/' + fileBasename + '_radCal.til'
            rrsRaster = e.OpenRaster(rrsRasterFile)
            ; Define an empty dictionary and an empty list to populate with results for each band
            rrsAggregator = Dictionary()
            rrsList = List()
            ; Extract each band from rrsRaster
            Task_ExtractBands = ENVITask('ExtractBandsFromRaster')
            Task_ExtractBands.INPUT_RASTER = rrsRaster
            Task_ExtractBands.Execute
            ; Loop through each band and apply band math
            FOREACH k, Task_ExtractBands.OUTPUT_RASTERS, k_index DO BEGIN
              ; Apply band math to divide all pixel values by pi
              Task_BandMath = ENVITask('PixelwiseBandMathRaster')
              Task_BandMath.INPUT_RASTER = k
              Task_BandMath.EXPRESSION = 'b1 / !pi'
              Task_BandMath.Execute
              ; Add results to rrsList
              rrsList.Add, Task_BandMath.OUTPUT_RASTER, /EXTRACT
              rrsAggregator.OUTPUT = rrsList
            ENDFOREACH
            ; Stack each band back together using BuildBandStack
            Task_Stack = ENVITask('BuildBandStack')
            Task_Stack.INPUT_RASTERS = rrsAggregator.OUTPUT
            Task_Stack.OUTPUT_RASTER_URI = "*"
            Task_Stack.Execute
            bandMathOutput = Task_Stack.OUTPUT_RASTER
            ; Transfer metadata from rrsRaster to bandMathOutput
            FOR metadataTag = 0, N_ELEMENTS(rrsRaster.Metadata.Tags) - 1 DO BEGIN
              IF bandMathOutput.Metadata.HasTag (rrsRaster.Metadata.Tags[metadataTag]) THEN BEGIN
                bandMathOutput.Metadata.UpdateItem, rrsRaster.Metadata.Tags[metadataTag], rrsRaster.Metadata[rrsRaster.Metadata.Tags[metadataTag]]
              ENDIF ELSE BEGIN
                bandMathOutput.Metadata.AddItem, rrsRaster.Metadata.Tags[metadataTag], rrsRaster.Metadata[rrsRaster.Metadata.Tags[metadataTag]]
              ENDELSE
            END
            ; Output results into new folder
            FILE_MKDIR, FILEPATH('4_R_To_Rrs/' + fileBasename + '_rToRrs/', ROOT_DIR = workingDir)
            bandMathOutput.Export, FILEPATH('4_R_To_Rrs/' + fileBasename + '_rToRrs/' + fileBasename + '_rToRrs.til', ROOT_DIR = workingDir), 'ENVI'
          ENDIF


          ; Get just water pixels by masking the raster by estuary ROI to subset spatially and masking by NDWI values to subset spectrally
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Masking raster image to just water pixels (Step 5 of 10)'
          IF ~ FILE_TEST(FILEPATH('5_Mask_Raster/' + fileBasename + '_maskRaster/' + fileBasename + '_maskRaster.til', ROOT_DIR = workingDir)) THEN BEGIN
            ; Read in R to Rrs result
            shpMaskRasterFile = FILEPATH('4_R_To_Rrs/' + fileBasename + '_rToRrs/' + fileBasename + '_rToRrs.til', ROOT_DIR = workingDir)
            shpMaskRasterInput = e.OpenRaster(shpMaskRasterFile)
            ; Mask shpMaskRasterInput by estuariesRoi to subset the scene spatially
            shpMaskRaster = ENVIROIMaskRaster(shpMaskRasterInput, estuariesRoi)
            ; Calculate NDWI on shpMaskRaster to subset the scene spectrally
            Task_NDWI = ENVITask('PixelwiseBandMathRaster')
            Task_NDWI.INPUT_RASTER = shpMaskRaster
            Task_NDWI.EXPRESSION = 'float(b2 - b5) / float(b2 + b5)'
            Task_NDWI.Execute
            rasterNDWI = Task_NDWI.OUTPUT_RASTER
            ; Create a mask raster based on an NDWI threshold
            ndwiThreshold = [0]
            ndwiThresholdRaster = ENVIBinaryLTThresholdRaster(rasterNDWI, ndwiThreshold)
            ; Apply ndwiThresholdRaster as a threshold mask on shpMaskRaster
            Task_Mask = ENVITask('MaskRaster')
            Task_Mask.DATA_IGNORE_VALUE = 999999.0
            Task_Mask.INPUT_MASK_RASTER = ndwiThresholdRaster
            Task_Mask.INPUT_RASTER = shpMaskRaster
            Task_Mask.INVERSE = 'TRUE'
            Task_Mask.Execute
            ; Output results into new folder
            maskOutput = Task_Mask.OUTPUT_RASTER
            FILE_MKDIR, FILEPATH('5_Mask_Raster/' + fileBasename + '_maskRaster/', ROOT_DIR = workingDir)
            maskOutput.Export, FILEPATH('5_Mask_Raster/' + fileBasename + '_maskRaster/' + fileBasename + '_maskRaster.til', ROOT_DIR = workingDir), 'ENVI'
          ENDIF


          ; Extract band four (red edge) values and save to list
          PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Computing atmospheric correction reference value (Step 6 of 10)'
          ; Read in results from 5_Mask_Raster
          redEdgeFile = FILEPATH('5_Mask_Raster/' + fileBasename + '_maskRaster/' + fileBasename + '_maskRaster.til', ROOT_DIR = workingDir)
          redEdgeInput = e.OpenRaster(redEdgeFile, DATA_IGNORE_VALUE = 999999.0)
          ; Subset to red edge values and get data greater than 999999 (our data ignore value)
          redEdgeValues = redEdgeInput.GetData(BANDS = [3])
          redEdgeValuesSubset = redEdgeValues[WHERE(redEdgeValues LT 999999.0)]
          redEdgeValuesSort = redEdgeValuesSubset[SORT(redEdgeValuesSubset)]
          ; Retrieve half of the lowest 5% of the data within the red edge
          redEdgeAnchor = MEDIAN(redEdgeValuesSort[0:(N_ELEMENTS(redEdgeValuesSubset) / 20) - 1]) / 2
  
  
          ; Add image information to their respective arrays
          tileFilename[j] = fileBasename
          tileViewAngle[j] = FLOAT(STRMID(metaInfo[1], STREGEX(metaInfo[1], 'view_angle') + 13, 8))
          tileSunAngle[j] = FLOAT(STRMID(metaInfo[1], STREGEX(metaInfo[1], 'sun_elevation') + 16, 13))
          tileREdgeAnchor[j] = redEdgeAnchor
          tileRayleigh[j] = rayleighExp
          tileProcessingTime[j] = SYSTIME()


        ENDELSE

      ENDELSE

    ENDFOR


    ; Save scene information as a CSV file
    PRINT, STRING(9B) + STRING(9B) + STRING(9B) + 'Outputing file information as a CSV file (Step 7 of 10)'
    IF ~ FILE_TEST(FILEPATH('File_Information/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '_fileInformation.csv', ROOT_DIR = workingDir)) THEN BEGIN
      ; Create a CSV file of the results
      FILE_MKDIR, FILEPATH('File_Information/', ROOT_DIR = workingDir)
      fileinfo_output = FILEPATH('File_Information/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '_fileInformation.csv', ROOT_DIR = workingDir)
      WRITE_CSV, fileinfo_output, tileFilename, tileViewAngle, tileSunAngle, tileREdgeAnchor, tileRayleigh, tileProcessingTime, $
        HEADER = ['FILENAME', 'VIEWANGLE', 'SUNANGLE', 'REDEDGEANCHOR', 'RAYLEIGH', 'PROCESSINGTIME']
    ENDIF


    ; Find the minimum rededge value for the entire overpass and compute the atmospheric correction values
    ; Define wavelengths
    wavelengths = [475,555,657.5,710,805]
    ; Compute the scattering factor based on the minRededge value
    minRedEdgeAnchor = MIN(tileREdgeAnchor)
    scatteringFactor = wavelengths[3] ^ rayleighExp * minRedEdgeAnchor
    ; Use the scatteringFactor to compute the correction values for each band
    atmCorrValues = MAKE_ARRAY(N_ELEMENTS(wavelengths), 1, /FLOAT, VALUE = -9999)
    FOR j = 0, N_ELEMENTS(wavelengths) - 1 DO BEGIN
      atmCorrValues[j] = scatteringFactor / wavelengths[j] ^ rayleighExp
    ENDFOR
  
  
    ; List processed files from 4_R_To_Rrs for further processing
    rToRrs_files = FILE_SEARCH(workingDir + '\4_R_To_Rrs\', '*.TIL')
  
  
    ; Loop through each tile in rToRrs_files and apply additional processing
    FOR j = 0, N_ELEMENTS(rToRrs_files) - 1 DO BEGIN
  
  
      ; Update the progress of the loop
      PRINT, STRING(9B) + STRING(9B) + 'Processing step 8 through 10 for tile' + STRCOMPRESS(STRING(j + 1) + ' of ' + STRING(N_ELEMENTS(satFile)))
  
  
      ; Define fileBasename
      fileBasename = STRMID(FILE_BASENAME(rToRrs_files[j],'.TIL'),0,STRLEN(FILE_BASENAME(rToRrs_files[j],'.TIL'))-7)
  
  
      ; Apply atmospheric correction
      PRINT, STRING(9B) + STRING(9B) + STRING(9B) + STRCOMPRESS('Applying atmospheric correction (Step 8 of 10)')
      IF ~ FILE_TEST(FILEPATH('6_Atm_Corr/' + fileBasename + '_atmCorr/' + fileBasename + '_atmCorr.til', ROOT_DIR = workingDir)) THEN BEGIN
        ; Read in raster results from 4_R_To_Rrs as atmospheric correction input
        atmCorrFile = rToRrs_files[j]
        atmCorrInput = e.OpenRaster(rToRrs_files[j])
        ; Apply the DarkSubtractionCorrection task
        Task_atmCorr = ENVITask('DarkSubtractionCorrection')
        Task_atmCorr.INPUT_RASTER = atmCorrInput
        Task_atmCorr.VALUES = atmCorrValues
        Task_atmCorr.Execute
        ; Output results into new folder
        atmCorrOutput = Task_atmCorr.OUTPUT_RASTER
        FILE_MKDIR, FILEPATH('6_Atm_Corr/' + fileBasename + '_atmCorr/', ROOT_DIR = workingDir)
        atmCorrOutput.Export, FILEPATH('6_Atm_Corr/' + fileBasename + '_atmCorr/' + fileBasename + '_atmCorr.til', ROOT_DIR = workingDir), 'ENVI'
      ENDIF


;    ; OPTIONAL -- Apply RPCOrthorectification
;    PRINT, STRING(9B) + STRING(9B) + STRING(9B) + STRCOMPRESS('Applying RPC Orthorectification (Step 9 of 10)')
;    IF ~ FILE_TEST(FILEPATH('7_RPC_Ortho/' + fileBasename + '_rpcOrtho/' + fileBasename + '_rpcOrtho.til', ROOT_DIR = workingDir)) THEN BEGIN
;      ; Read in results from atmospheric correction
;      rpcOrthoFile = FILEPATH('6_Atm_Corr/' + fileBasename + '_atmCorr/' + fileBasename + '_atmCorr.til', ROOT_DIR = workingDir)
;      rpcOrthoInput = e.OpenRaster(rpcOrthoFile, DATA_IGNORE_VALUE = 999999)
;      ; Apply the RPCOrthorectification task
;      Task_rpcOrtho = ENVITask('RPCOrthorectification')
;      Task_rpcOrtho.INPUT_RASTER = rpcOrthoInput
;      Task_rpcOrtho.DEM_RASTER = demInput
;      Task_rpcOrtho.OUTPUT_PIXEL_SIZE = [6.5,6.5]
;      Task_rpcOrtho.Execute
;      ; Output results into new folder
;      rpcOrthoOutput = Task_rpcOrtho.OUTPUT_RASTER
;      FILE_MKDIR, FILEPATH('7_RPC_Ortho/' + fileBasename + '_rpcOrtho/', ROOT_DIR = workingDir)
;      rpcOrthoOutput.Export, FILEPATH('7_RPC_Ortho/' + fileBasename + '_rpcOrtho/' + fileBasename + '_rpcOrtho.til', ROOT_DIR = workingDir), 'ENVI'
;    ENDIF

  ENDFOR


;    ; OPTIONAL -- Apply mosaicing
;    PRINT, STRING(9B) + STRING(9B) + STRCOMPRESS('Mosaicing tiles into single scene (Step 10 of 10)')
;    IF ~ FILE_TEST(FILEPATH('8_Mosaic/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '_mosaic/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '_mosaic.til', ROOT_DIR = workingDir)) THEN BEGIN
;      ; List RPC Orthorectified rasters
;      rpcOrthoList = FILE_SEARCH(FILEPATH('7_RPC_Ortho/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '*_rpcOrtho', ROOT_DIR = workingDir), '*rpcOrtho.til', /FULLY_QUALIFY_PATH)
;      ; Create an empty list to populate with rasters
;      rpcOrthoRasters = List()
;      ; Loop through each raster, read it in, and add to rpcOrthoRasters
;      FOR k = 0, N_ELEMENTS(rpcOrthoList) - 1 DO BEGIN
;        rpcOrthoRaster = e.OpenRaster(rpcOrthoList[k])
;        rpcOrthoRasters.Add, rpcOrthoRaster
;      ENDFOR
;      ; Convert rpcOrthoRasters to an array
;      mosaicInputRasters = rpcOrthoRasters.ToArray()
;      ; Apply the BuildMosaicRaster task
;      Task_mosaic = ENVITask('BuildMosaicRaster')
;      Task_mosaic.INPUT_RASTERS = mosaicInputRasters
;      Task_mosaic.Execute
;      ; Output results into new folder
;      mosaicOutput = Task_mosaic.OUTPUT_RASTER
;      FILE_MKDIR, FILEPATH('8_Mosaic/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '_mosaic/', ROOT_DIR = workingDir)
;      mosaicOutput.Export, FILEPATH('8_Mosaic/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '_mosaic/' + STRMID(fileBasename, 0, STRLEN(fileBasename) - 2) + '_mosaic.til', ROOT_DIR = workingDir), 'ENVI'
;    ENDIF

  ENDFOR
  
  ; Close ENVI and delete temporary folder
  e.Close
  ;FILE_DELETE, tempFolder, /RECURSIVE

END

